"use client";
import { useEffect, useMemo, useState } from "react";
import axios from "axios";
import dynamic from "next/dynamic";

type Site = { id: string; name: string; centroid?: {lat:number; lng:number} | null };
const Map = dynamic(() => import("./osm-map"), { ssr: false });

type Hazard = { id: string; title: string; severity: string; status: string; pin: {lat:number; lng:number} };

export default function Dashboard() {
  const api = process.env.NEXT_PUBLIC_API_BASE || "http://localhost:8000";

  const [sites, setSites] = useState<Site[]>([]);
  const [siteId, setSiteId] = useState<string>("");
  const [hazards, setHazards] = useState<Hazard[]>([]);
  const [newSiteName, setNewSiteName] = useState("Demo Site");
  const [uploadFile, setUploadFile] = useState<File | null>(null);

  const token = typeof window !== "undefined" ? localStorage.getItem("token") : null;

  useEffect(() => {
    if (!token) window.location.href = "/app/login";
  }, [token]);

  const headers = useMemo(() => ({ Authorization: `Bearer ${token}` }), [token]);

  async function loadSites() {
    const r = await axios.get(`${api}/sites`, { headers });
    setSites(r.data);
    if (r.data?.length && !siteId) setSiteId(r.data[0].id);
  }

  async function loadHazards(sid: string) {
    const r = await axios.get(`${api}/sites/${sid}/hazards`, { headers });
    setHazards(r.data);
  }

  useEffect(() => { if (token) loadSites(); }, [token]);
  useEffect(() => { if (siteId) loadHazards(siteId); }, [siteId]);

  async function createSite() {
    const r = await axios.post(`${api}/sites`, { name: newSiteName }, { headers });
    await loadSites();
    setSiteId(r.data.id);
  }

  async function uploadMedia() {
    if (!uploadFile || !siteId) return;
    // Step 1: request presigned URL
    const create = await axios.post(`${api}/media`, {
      site_id: siteId,
      media_type: uploadFile.type.startsWith("video/") ? "VIDEO" : "PHOTO",
      filename: uploadFile.name,
      content_type: uploadFile.type,
      blur_faces: true
    }, { headers });

    const { id, upload_url } = create.data;
    // Step 2: upload to MinIO via presigned URL
    await axios.put(upload_url, uploadFile, { headers: { "Content-Type": uploadFile.type } });
    // Step 3: finalize (enqueue worker)
    
    alert("Uploaded. AI job queued. Refresh hazards in ~10-30s.");
  }

  const center = sites.find(s => s.id === siteId)?.centroid || { lat: 45.5019, lng: -73.5674 };


  return (
    <main style={{ display: "grid", gridTemplateColumns: "340px 1fr", height: "100vh" }}>
      <aside style={{ padding: 16, borderRight: "1px solid #eee", overflow: "auto" }}>
        <h2 style={{ marginTop: 0 }}>Dashboard</h2>

        <div style={{ marginBottom: 16 }}>
          <label>Site</label>
          <select value={siteId} onChange={(e)=>setSiteId(e.target.value)} style={{ width: "100%", padding: 8 }}>
            {sites.map(s => <option key={s.id} value={s.id}>{s.name}</option>)}
          </select>
        </div>

        <div style={{ padding: 12, border: "1px solid #eee", borderRadius: 10, marginBottom: 16 }}>
          <h3 style={{ marginTop: 0 }}>Create site</h3>
          <input value={newSiteName} onChange={(e)=>setNewSiteName(e.target.value)} style={{ width: "100%", padding: 8 }} />
          <button onClick={createSite} style={{ marginTop: 8, width: "100%" }}>Create</button>
        </div>

        <div style={{ padding: 12, border: "1px solid #eee", borderRadius: 10, marginBottom: 16 }}>
          <h3 style={{ marginTop: 0 }}>Upload media</h3>
          <input type="file" accept="image/*,video/*" onChange={(e)=>setUploadFile(e.target.files?.[0] || null)} />
          <button onClick={uploadMedia} style={{ marginTop: 8, width: "100%" }} disabled={!uploadFile}>Upload & Process</button>
        </div>

        <div style={{ display: "flex", gap: 8 }}>
          <button onClick={()=>loadHazards(siteId)} style={{ flex: 1 }}>Refresh hazards</button>
          <button onClick={()=>{ localStorage.removeItem("token"); window.location.href="/app/login";}} style={{ flex: 1 }}>Logout</button>
        </div>

        <h3 style={{ marginTop: 18 }}>Hazards</h3>
        <div style={{ display: "grid", gap: 8 }}>
          {hazards.map(h => (
            <div key={h.id} style={{ border: "1px solid #eee", borderRadius: 10, padding: 10 }}>
              <div style={{ display: "flex", justifyContent: "space-between" }}>
                <b>{h.title}</b>
                <span>{h.severity}</span>
              </div>
              <div style={{ color: "#666" }}>{h.status}</div>
              <div style={{ fontSize: 12, color: "#888" }}>{h.pin.lat.toFixed(5)}, {h.pin.lng.toFixed(5)}</div>
            </div>
          ))}
          {!hazards.length && <p style={{ color: "#666" }}>No hazards yet. Upload a snowy photo to see WINTER hazards.</p>}
        </div>
      </aside>

      <section style={{ position: "relative" }}>
        {!gkey && (
          <div style={{ position: "absolute", zIndex: 2, top: 12, left: 12, background: "rgba(255,255,255,0.95)", padding: 12, borderRadius: 10, border: "1px solid #eee" }}>
            <b>Google Maps key missing.</b>
            <div>Add <code>GOOGLE_MAPS_API_KEY</code> in <code>.env</code> then restart.</div>
          </div>
        )}

        {<Map center={center} hazards={hazards} />}
      </section>
    </main>
  );
}
