from __future__ import annotations
from sqlalchemy import String, Text, Boolean, ForeignKey, DateTime, func, JSON, Float, Integer
from sqlalchemy.orm import Mapped, mapped_column, relationship
from geoalchemy2 import Geography, Geometry
from .db import Base
import uuid

def _uuid():
    return str(uuid.uuid4())

class Org(Base):
    __tablename__ = "orgs"
    id: Mapped[str] = mapped_column(String, primary_key=True, default=_uuid)
    name: Mapped[str] = mapped_column(String, nullable=False)
    settings: Mapped[dict] = mapped_column(JSON, default=dict)
    created_at: Mapped[DateTime] = mapped_column(DateTime(timezone=True), server_default=func.now())

class User(Base):
    __tablename__ = "users"
    id: Mapped[str] = mapped_column(String, primary_key=True, default=_uuid)
    org_id: Mapped[str] = mapped_column(String, ForeignKey("orgs.id", ondelete="CASCADE"), nullable=False)
    email: Mapped[str] = mapped_column(String, unique=True, nullable=False)
    pw_hash: Mapped[str] = mapped_column(String, nullable=False)
    role: Mapped[str] = mapped_column(String, nullable=False)
    is_active: Mapped[bool] = mapped_column(Boolean, default=True)
    created_at: Mapped[DateTime] = mapped_column(DateTime(timezone=True), server_default=func.now())

class Site(Base):
    __tablename__ = "sites"
    id: Mapped[str] = mapped_column(String, primary_key=True, default=_uuid)
    org_id: Mapped[str] = mapped_column(String, ForeignKey("orgs.id", ondelete="CASCADE"), nullable=False)
    name: Mapped[str] = mapped_column(String, nullable=False)
    address: Mapped[str | None] = mapped_column(String, nullable=True)
    boundary: Mapped[str | None] = mapped_column(Geometry("POLYGON", srid=4326), nullable=True)
    centroid: Mapped[str | None] = mapped_column(Geography("POINT", srid=4326), nullable=True)
    created_at: Mapped[DateTime] = mapped_column(DateTime(timezone=True), server_default=func.now())

class Zone(Base):
    __tablename__ = "zones"
    id: Mapped[str] = mapped_column(String, primary_key=True, default=_uuid)
    site_id: Mapped[str] = mapped_column(String, ForeignKey("sites.id", ondelete="CASCADE"), nullable=False)
    name: Mapped[str] = mapped_column(String, nullable=False)
    polygon: Mapped[str] = mapped_column(Geometry("POLYGON", srid=4326), nullable=False)
    created_at: Mapped[DateTime] = mapped_column(DateTime(timezone=True), server_default=func.now())

class Media(Base):
    __tablename__ = "media"
    id: Mapped[str] = mapped_column(String, primary_key=True, default=_uuid)
    site_id: Mapped[str] = mapped_column(String, ForeignKey("sites.id", ondelete="CASCADE"), nullable=False)
    uploader_id: Mapped[str] = mapped_column(String, ForeignKey("users.id"), nullable=False)
    media_type: Mapped[str] = mapped_column(String, nullable=False)  # PHOTO/VIDEO/DRONE_VIDEO/ORTHOMOSAIC
    s3_key: Mapped[str] = mapped_column(String, nullable=False)
    thumb_s3_key: Mapped[str | None] = mapped_column(String, nullable=True)
    captured_at: Mapped[DateTime | None] = mapped_column(DateTime(timezone=True), nullable=True)
    gps: Mapped[str | None] = mapped_column(Geography("POINT", srid=4326), nullable=True)
    heading: Mapped[float | None] = mapped_column(Float, nullable=True)
    altitude: Mapped[float | None] = mapped_column(Float, nullable=True)
    device_meta: Mapped[dict] = mapped_column(JSON, default=dict)
    blur_faces_applied: Mapped[bool] = mapped_column(Boolean, default=False)
    created_at: Mapped[DateTime] = mapped_column(DateTime(timezone=True), server_default=func.now())

class Detection(Base):
    __tablename__ = "detections"
    id: Mapped[str] = mapped_column(String, primary_key=True, default=_uuid)
    media_id: Mapped[str] = mapped_column(String, ForeignKey("media.id", ondelete="CASCADE"), nullable=False)
    frame_time_ms: Mapped[int | None] = mapped_column(Integer, nullable=True)
    hazard_signal: Mapped[str] = mapped_column(String, nullable=False)
    confidence: Mapped[float] = mapped_column(Float, nullable=False)
    severity: Mapped[str] = mapped_column(String, nullable=False)
    bbox: Mapped[dict | None] = mapped_column(JSON, nullable=True)
    polygon: Mapped[dict | None] = mapped_column(JSON, nullable=True)
    rule_tags: Mapped[list] = mapped_column(JSON, default=list)
    pin: Mapped[str | None] = mapped_column(Geography("POINT", srid=4326), nullable=True)
    created_at: Mapped[DateTime] = mapped_column(DateTime(timezone=True), server_default=func.now())

class Hazard(Base):
    __tablename__ = "hazards"
    id: Mapped[str] = mapped_column(String, primary_key=True, default=_uuid)
    site_id: Mapped[str] = mapped_column(String, ForeignKey("sites.id", ondelete="CASCADE"), nullable=False)
    status: Mapped[str] = mapped_column(String, nullable=False)  # OPEN/FIXED/VERIFIED/DISMISSED
    hazard_type: Mapped[str] = mapped_column(String, nullable=False)  # FALL/PPE/WINTER/etc.
    title: Mapped[str] = mapped_column(String, nullable=False)
    description: Mapped[str | None] = mapped_column(Text, nullable=True)
    severity: Mapped[str] = mapped_column(String, nullable=False)
    primary_rule_tags: Mapped[list] = mapped_column(JSON, default=list)
    pin: Mapped[str] = mapped_column(Geography("POINT", srid=4326), nullable=False)
    zone_id: Mapped[str | None] = mapped_column(String, ForeignKey("zones.id"), nullable=True)
    assignee_id: Mapped[str | None] = mapped_column(String, ForeignKey("users.id"), nullable=True)
    created_by: Mapped[str] = mapped_column(String, ForeignKey("users.id"), nullable=False)
    created_at: Mapped[DateTime] = mapped_column(DateTime(timezone=True), server_default=func.now())
    updated_at: Mapped[DateTime] = mapped_column(DateTime(timezone=True), server_default=func.now(), onupdate=func.now())

class HazardEvidence(Base):
    __tablename__ = "hazard_evidence"
    id: Mapped[str] = mapped_column(String, primary_key=True, default=_uuid)
    hazard_id: Mapped[str] = mapped_column(String, ForeignKey("hazards.id", ondelete="CASCADE"), nullable=False)
    media_id: Mapped[str] = mapped_column(String, ForeignKey("media.id", ondelete="CASCADE"), nullable=False)
    detection_id: Mapped[str | None] = mapped_column(String, ForeignKey("detections.id", ondelete="SET NULL"), nullable=True)
    note: Mapped[str | None] = mapped_column(Text, nullable=True)
    created_at: Mapped[DateTime] = mapped_column(DateTime(timezone=True), server_default=func.now())
