from __future__ import annotations
import io, os, subprocess, tempfile
from PIL import Image

def extract_frame_from_video(video_bytes: bytes, at_seconds: int = 1) -> Image.Image:
    # Use ffmpeg to extract a single frame to PNG
    with tempfile.TemporaryDirectory() as td:
        in_path = os.path.join(td, "in.mp4")
        out_path = os.path.join(td, "frame.png")
        with open(in_path, "wb") as f:
            f.write(video_bytes)
        cmd = [
            "ffmpeg", "-hide_banner", "-loglevel", "error",
            "-ss", str(at_seconds), "-i", in_path,
            "-frames:v", "1", out_path
        ]
        subprocess.check_call(cmd)
        img = Image.open(out_path).convert("RGB")
        return img

def snow_whiteness_ratio(img: Image.Image) -> float:
    # Simple heuristic: ratio of pixels near white.
    # Works surprisingly well for "snow-covered ground" but is not a true ice detector.
    im = img.resize((320, 180))
    px = im.getdata()
    white = 0
    total = 0
    for r,g,b in px:
        total += 1
        if r > 220 and g > 220 and b > 220:
            white += 1
    return white / max(total, 1)

def run_stub(img: Image.Image) -> list[dict]:
    # Returns list of detection dicts (signal, confidence, bbox optional)
    w_ratio = snow_whiteness_ratio(img)
    dets = []
    if w_ratio > 0.35:
        dets.append({"signal":"WINTER_SNOW_ON_WALKWAY","confidence": min(0.95, 0.6 + w_ratio)})
    elif w_ratio > 0.20:
        dets.append({"signal":"WINTER_ICE_SUSPECTED","confidence": 0.55})
    # Add a couple generic demo detections at low probability
    # (In production these come from real models)
    return dets
